/**
 * 
 */
package gov.va.med.mhv.usermgmt.service;

import java.util.Calendar;
import java.util.Date;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.tigris.atlas.service.ServiceResponse;

import gov.va.med.mhv.usermgmt.enumeration.BloodTypeEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.ContactMethodEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.CountryEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.GenderEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.MaritalStatusEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.PasswordHintQuestionEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.StateEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.TitleEnumeration;
import gov.va.med.mhv.usermgmt.messages.UserManagementMessages;
import gov.va.med.mhv.usermgmt.service.delegate.ServiceDelegateFactory;
import gov.va.med.mhv.usermgmt.transfer.TransferObjectFactory;
import gov.va.med.mhv.usermgmt.transfer.UserProfile;
import gov.va.med.mhv.usermgmt.util.AccountSecurityManager;

/**
 * @author Rob Proper
 *
 */
public class TestUserServiceAuthenticate extends BaseTestCase {
	
	static final Log LOG = LogFactory.getLog(
		TestUserServiceAuthenticate.class);
	
	private UserBuilder user = null;

	/**
	 * Constructor for UserServiceAuthenticateTest.
	 * @param arg0
	 */
	public TestUserServiceAuthenticate(String arg0) {
		super(arg0);
	}

	/*
	 * @see TestCase#tearDown()
	 */
	protected void tearDown() throws Exception {
		if (user != null) {
			user.tearDown();
		}
	}
	
	public void testMissingUsername() throws Exception {
		assertHasError(ServiceDelegateFactory.createUserServiceDelegate().
			authenticate("", "donotcare"),
			UserManagementMessages.USERNAME_PASSWORD_REQUIRED);
	}
	
	public void testMissingPassword() throws Exception {
		assertHasError(ServiceDelegateFactory.createUserServiceDelegate().
			authenticate(UserBuilder.generateUserName(), ""),
			UserManagementMessages.USERNAME_PASSWORD_REQUIRED);
	}

	public void testInvalidCredentialsUnknownUserName() throws Exception {
		assertHasError(ServiceDelegateFactory.createUserServiceDelegate().
			authenticate(UserBuilder.generateUserName(), "donnotcare"),
			UserManagementMessages.INVALID_CREDENTIALS);
	}

	public void testInvalidCredentialsInvalidPassword() throws Exception {
		setUpTestUser();
		assertHasError(ServiceDelegateFactory.createUserServiceDelegate().
			authenticate(user.getUserName(), user.createInvalidPassword()),
			UserManagementMessages.INVALID_CREDENTIALS);
	}

	public void testValidCredentials() throws Exception {
		setUpTestUser();
		assertNoErrors(ServiceDelegateFactory.
			createUserServiceDelegate().authenticate(user.getUserName(), 
				user.getPassword()));
	}

	public void testAccountNotLockingBelowThreshold() throws Exception {
		setUpTestUser();
		int n = AccountSecurityManager.getInstance().getLockoutThreshold();
		forceAuthenticationFailures(n-1);
		ServiceResponse response = ServiceDelegateFactory.
			createUserServiceDelegate().authenticate(user.getUserName(), 
			user.getPassword());
		assertNoErrors("\n\t" + n + "-th authentication attempt should succeed, "
			+ "when valid password and number of allowed subsequent failures = [" 
			+ n + "]", response);
		assertFalse("Account should not be locked after " + n + " failed "
			+ "authentication attempts and number of allowed subsequent "
			+ "failures = [" + n + "]", ServiceDelegateFactory.
			createUserServiceDelegate().isAccountLocked(user.getUserName()).
			getBoolean().booleanValue());
	}

	public void testAccountLockingUponThreshold() throws Exception {
		setUpTestUser();
		int n = AccountSecurityManager.getInstance().getLockoutThreshold();
		forceAuthenticationFailures(n-1);
		ServiceResponse response = ServiceDelegateFactory.
			createUserServiceDelegate().authenticate(user.getUserName(), 
			user.createInvalidPassword());
		assertHasError("\n\t" + n + "-th authentication attempt should fail, "
			+ " when invalid password and number of allowed subsequent "
			+ "failures = [" + n + "]", 
			response, UserManagementMessages.ACCOUNT_LOCKED);
	}

	public void testAccountLockingOverThreshold() throws Exception {
		setUpTestUser();
		int n = AccountSecurityManager.getInstance().getLockoutThreshold();
		forceAuthenticationFailures(n);
		ServiceResponse response = ServiceDelegateFactory.
			createUserServiceDelegate().authenticate(user.getUserName(), 
			user.createInvalidPassword());
		assertHasError("\n\t(" + (n+1) + ")-th authentication attempt should fail,"
			+ " when number of allowed subsequent failures = [" + n + "]", 
			response, UserManagementMessages.ACCOUNT_LOCKED);
		assertTrue("Account must be locked after " + (n+1) + " failed "
			+ "authentication attempts and number of allowed subsequent "
			+ "failures = [" + n + "]", ServiceDelegateFactory.
			createUserServiceDelegate().isAccountLocked(user.getUserName()).
			getBoolean().booleanValue());
	}

	public void testAccountUnlockedAfterPasswordReset() throws Exception {
		testAccountLockingUponThreshold();
		String newPassword = user.createResetPassword(); 
		ServiceResponse response = ServiceDelegateFactory.
			createUserServiceDelegate().resetPassword(user.getUserName(), 
				newPassword, newPassword, false);
		assertNoErrors("\n\tFail to reset password to '" + newPassword + "'",
			response);
		response = ServiceDelegateFactory.createUserServiceDelegate().
			authenticate(user.getUserName(), newPassword);
		assertNoErrors("\n\tauthentication attempt should succeed, after reset",
			response);
		assertFalse("Account should not be locked after password reset", 
			ServiceDelegateFactory.createUserServiceDelegate().isAccountLocked(
			user.getUserName()).getBoolean().booleanValue());
	}

	public void testCannotChangePasswordWhenAccountLocked() throws Exception {
		testAccountLockingUponThreshold();
		String newPassword = user.createResetPassword(); 
		ServiceResponse response = ServiceDelegateFactory.
			createUserServiceDelegate().changePassword(user.getUserName(), 
				user.getPassword(), newPassword, newPassword);
		assertHasError("\n\tFail to reset password to '" + newPassword + "'",
			response, UserManagementMessages.ACCOUNT_LOCKED);
	}


	private void setUpTestUser() throws Exception {
		user = new UserBuilder(createUserProfile(null, "John", "Doe", 
			UserBuilder.createDate(Calendar.JANUARY, 1, 1970)), "%password0");
		assertNotNull("Failed to set up test user", user);
		user.setUp();
	}

	private static UserProfile createUserProfile(String userName, 
		String firstName, String lastName, Date birthDate) 
	{
		UserProfile userProfile = TransferObjectFactory.createUserProfile();
		userProfile.setUserName(userName);
		userProfile.setFirstName(firstName);
		userProfile.setLastName(lastName);
		userProfile.setMiddleName("");
		userProfile.setBirthDate(birthDate);
		userProfile.setTitle(TitleEnumeration.getEnum(TitleEnumeration.MR));
		userProfile.setBloodType(BloodTypeEnumeration.getEnum(
			BloodTypeEnumeration.O_NEGATIVE));
		userProfile.setIsOrganDonor(Boolean.TRUE);
		userProfile.setGender(GenderEnumeration.getEnum(GenderEnumeration.MALE));
		userProfile.setMaritalStatus(MaritalStatusEnumeration.getEnum(
			MaritalStatusEnumeration.DIVORCED));
		userProfile.setIsVeteran(Boolean.TRUE);
		userProfile.setIsPatient(false);
		userProfile.setAddressStreet1("1 Some Street");
		userProfile.setAddressCity("Nowhere Town");
		userProfile.setAddressState(StateEnumeration.getEnum(StateEnumeration.VA));
		userProfile.setAddressPostalCode("12345");
		userProfile.setAddressCountry(CountryEnumeration.getEnum(
			CountryEnumeration.UNITED_STATES));
		userProfile.setContactInfoContactMethod(ContactMethodEnumeration.getEnum(
			ContactMethodEnumeration.HOMEPHONE));
		userProfile.setContactInfoHomePhone("555-555-5555");
		userProfile.setContactInfoEmail(firstName + "." + lastName + 
			"serv.domain.ext");
		userProfile.setPasswordHintQuestion1(PasswordHintQuestionEnumeration.
			getEnum(PasswordHintQuestionEnumeration.FAVORITEFOOD));
		userProfile.setPasswordHintAnswer1("Pizza");
		userProfile.setPasswordHintQuestion2(PasswordHintQuestionEnumeration.
			getEnum(PasswordHintQuestionEnumeration.PETNAME));
		userProfile.setPasswordHintAnswer2("Spot");
		userProfile.setAcceptDisclaimer(Boolean.TRUE);
		userProfile.setAcceptPrivacy(Boolean.TRUE);
		userProfile.setAcceptTerms(Boolean.TRUE);
		return userProfile;
	}
	
	
	private void forceAuthenticationFailures(int n) {
		for (int i = 1; i <= n; i++) {
			// Fake authentication failures
			ServiceDelegateFactory.createUserServiceDelegate().
				authenticate(user.getUserName(), user.createInvalidPassword());
		}
	}

}
